<?php

namespace oms\userdb\controllers\backend;

use oms\userdb\models\UserSearch;
use Yii;
use oms\userdb\enums\Role;
use oms\userdb\models\User;
use oms\userdb\repositories\Repository;
use oms\userdb\repositories\UserRepository;
use oms\userdb\components\actions\FormAction;
use oms\userdb\models\UserForm;
use yii\filters\VerbFilter;
use yii\web\NotFoundHttpException;

class UserController extends Controller
{
    protected $repository;

    public function __construct($id, $module, UserRepository $repository, $config = [])
    {
        parent::__construct($id, $module, $config);
        
        $this->repository = $repository;
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        $behaviors = parent::behaviors();

        $behaviors['verbs'] = [
            'class' => VerbFilter::className(),
            'actions' => [
                'resend-email' => ['post'],
            ]
        ];

        return $behaviors;
    }
    
    public function actions()
    {
        $formAction = [
            'class' => FormAction::className(),
            'formClass' => UserForm::className(),
            'repository' => $this->repository,
        ];
        
        return [
            'create' => $formAction + [
                'model' => function () {
                    return new User(['role' => Role::USER]);
                },
            ],
            'update' => $formAction + [
                'model' => function ($id) {
                    return $this->findModel($id, 'groupIds');
                },
            ],
        ];
    }
    
    
    /**
     * Lists all User models.
     * @return mixed
     */
    public function actionIndex($page = null)
    {
        $searchModel = new UserSearch();
        $searchModel->load(Yii::$app->request->queryParams);
        $dataProvider = $this->repository->provider($page, 'hasPassword,lastEmailStatus,lastVisit,loginAttempts,status');
        return $this->render('index', [
            'dataProvider' => $dataProvider,
        ]);
    }
    
    /**
     * Lists users access status.
     * @return mixed
     */
    public function actionAccess($page = null)
    {
        $dataProvider = $this->repository->provider($page, 'hasPassword,lastEmailStatus,lastVisit,loginAttempts');
        return $this->render('access', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single User model.
     * @param integer $id
     * @return mixed
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id, 'groups, passwordResetLink, passwordResetExpireTime'),
        ]);
    }

    /**
     * Send new email to user with 'Password Reset Request Form'
     * @param $id
     * @param null $page
     * @return \yii\web\Response
     */
    public function actionResendEmail($id, $page = null)
    {
        $resp = $this->repository->resendEmail($id);
        Yii::$app->session->setFlash($resp ? 'success' : 'error', 'with email sending');

        return $this->redirect(['access', 'page' => $page]);
    }
    
    /**
     * Refreshes user password reset token.
     * @param $id
     * @return \yii\web\Response
     */
    public function actionRefreshToken($id)
    {
        $this->repository->refreshToken($id);

        return $this->redirect(['view', 'id' => $id]);
    }

    /**
     * Deletes an existing User model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    /*public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }*/

    /**
     * Finds the User model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return User the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id, $fields = null)
    {
        $model = $this->repository->view($id, $fields);
        
        if ($model) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}
