<?php

namespace oms\userapi;

use Assert\Assertion;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Exception\ClientException;
use oms\userapi\data\Response;

class Client
{
    private $baseUrl;
    
    /**
     *
     * @var data\Auth
     */
    private $auth;
    
    /**
     *
     * @var ClientInterface 
     */
    private $httpClient = null;
    
    public function __construct($baseUrl, data\Auth $auth, ClientInterface $httpClient)
    {
        $this->baseUrl = $baseUrl;
        $this->auth = $auth;
        $this->httpClient = $httpClient;
    }
    
    /**
     * 
     * @param string $method
     * @param string $uri
     * @param type $body
     * @return data\Response
     */
    public function request($method, $uri, $body = null)
    {
        Assertion::inArray($method, ['GET', 'POST']);
        Assertion::string($uri);
        
        $url = $this->baseUrl . $uri;
        $headers = [
            'Content-Type' => 'application/json',
        ];
        $body = isset($body) ? json_encode($body) : '';
        $request = new Request($method, $url, $headers, $body);
        return $this->send($request);
    }
    
    /**
     * 
     * @param Request $request
     * @return data\Response
     */
    public function send(Request $request)
    {
        $fields = new data\AuthorizationHeaderFields($this->auth);
        $header = new logic\AuthorizationHeader($fields, $request);
        
        $signedRequest = $header->sign($request);
        
        try {
            $rawResponse = $this->httpClient->send($signedRequest, [
                'allow_redirects' => false,
            ]);

        } catch (ClientException $e) {
            if ($e->getResponse()->hasHeader(Response::HEADER_TIMESTAMP)) {
                $serverTimestamp = (int) $e->getResponse()->getHeaderLine(Response::HEADER_TIMESTAMP);
                $this->auth->setTimestamp($serverTimestamp);
                return $this->send($request);
            }
            
            throw $e;
        }
        
        if ($rawResponse->hasHeader(Response::HEADER_USER_TOKEN)) {
            $this->auth->setUserToken($rawResponse->getHeaderLine(Response::HEADER_USER_TOKEN));
        }

        $response = new Response($rawResponse);

        return $response;
    }
    
    /**
     * 
     * @return data\Auth
     */
    public function getAuth()
    {
        return $this->auth;
    }
}
